// Database Module using IndexedDB
const db = {
    dbName: 'PokerTrackerDB',
    version: 1,
    db: null,
    
    // Initialize the database
    init() {
        return new Promise((resolve, reject) => {
            const request = indexedDB.open(this.dbName, this.version);
            
            request.onerror = () => {
                console.error('Database failed to open');
                reject(request.error);
            };
            
            request.onsuccess = () => {
                this.db = request.result;
                console.log('Database opened successfully');
                resolve(this.db);
            };
            
            request.onupgradeneeded = (e) => {
                this.db = e.target.result;
                console.log('Database upgrade needed');
                
                // Create object stores
                if (!this.db.objectStoreNames.contains('games')) {
                    const gamesStore = this.db.createObjectStore('games', { keyPath: 'id' });
                    gamesStore.createIndex('status', 'status', { unique: false });
                    gamesStore.createIndex('createdDate', 'createdDate', { unique: false });
                }
                
                if (!this.db.objectStoreNames.contains('players')) {
                    const playersStore = this.db.createObjectStore('players', { keyPath: 'id' });
                    playersStore.createIndex('gameId', 'gameId', { unique: false });
                    playersStore.createIndex('name', 'name', { unique: false });
                }
                
                if (!this.db.objectStoreNames.contains('roster')) {
                    const rosterStore = this.db.createObjectStore('roster', { keyPath: 'id' });
                    rosterStore.createIndex('name', 'name', { unique: false });
                }
                
                if (!this.db.objectStoreNames.contains('chipCounts')) {
                    const chipCountsStore = this.db.createObjectStore('chipCounts', { 
                        keyPath: 'id', 
                        autoIncrement: true 
                    });
                    chipCountsStore.createIndex('gameId', 'gameId', { unique: false });
                    chipCountsStore.createIndex('playerId', 'playerId', { unique: false });
                }
                
                if (!this.db.objectStoreNames.contains('settings')) {
                    this.db.createObjectStore('settings', { keyPath: 'key' });
                }
                
                console.log('Database setup complete');
            };
        });
    },
    
    // Game operations
    saveGame(game) {
        return this.put('games', game);
    },
    
    getGame(gameId) {
        return this.get('games', gameId);
    },
    
    getAllGames() {
        return this.getAll('games');
    },
    
    getPausedGame() {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction(['games'], 'readonly');
            const store = transaction.objectStore('games');
            const index = store.index('status');
            const request = index.get('paused');
            
            request.onsuccess = () => resolve(request.result);
            request.onerror = () => reject(request.error);
        });
    },
    
    updateGameStatus(gameId, status) {
        return this.getGame(gameId).then(game => {
            if (game) {
                game.status = status;
                if (status === 'completed') {
                    game.endedDate = new Date().toISOString();
                }
                return this.saveGame(game);
            }
        });
    },
    
    // Player operations
    savePlayers(gameId, players) {
        const promises = players.map(player => {
            player.gameId = gameId;
            return this.put('players', player);
        });
        return Promise.all(promises);
    },
    
    getGamePlayers(gameId) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction(['players'], 'readonly');
            const store = transaction.objectStore('players');
            const index = store.index('gameId');
            const request = index.getAll(gameId);
            
            request.onsuccess = () => resolve(request.result);
            request.onerror = () => reject(request.error);
        });
    },
    
    updatePlayer(player) {
        return this.put('players', player);
    },
    
    // Roster operations
    getAllPlayers() {
        return this.getAll('roster');
    },
    
    addToRoster(player) {
        if (!player.id) {
            player.id = this.generateGUID();
        }
        return this.put('roster', player);
    },
    
    removeFromRoster(playerId) {
        return this.delete('roster', playerId);
    },
    
    // Chip count operations
    saveChipCount(chipCount) {
        chipCount.timestamp = new Date().toISOString();
        return this.put('chipCounts', chipCount);
    },
    
    getChipCounts(gameId) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction(['chipCounts'], 'readonly');
            const store = transaction.objectStore('chipCounts');
            const index = store.index('gameId');
            const request = index.getAll(gameId);
            
            request.onsuccess = () => resolve(request.result);
            request.onerror = () => reject(request.error);
        });
    },
    
    getLatestChipCounts(gameId) {
        return this.getChipCounts(gameId).then(counts => {
            // Get the latest count for each player
            const latestCounts = {};
            counts.forEach(count => {
                if (!latestCounts[count.playerId] || 
                    new Date(count.timestamp) > new Date(latestCounts[count.playerId].timestamp)) {
                    latestCounts[count.playerId] = count;
                }
            });
            return Object.values(latestCounts);
        });
    },
    
    // Settings operations
    saveSetting(key, value) {
        return this.put('settings', { key, value });
    },
    
    getSetting(key) {
        return this.get('settings', key).then(result => result ? result.value : null);
    },
    
    // Generic database operations
    put(storeName, data) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readwrite');
            const store = transaction.objectStore(storeName);
            const request = store.put(data);
            
            request.onsuccess = () => resolve(request.result);
            request.onerror = () => reject(request.error);
        });
    },
    
    get(storeName, key) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readonly');
            const store = transaction.objectStore(storeName);
            const request = store.get(key);
            
            request.onsuccess = () => resolve(request.result);
            request.onerror = () => reject(request.error);
        });
    },
    
    getAll(storeName) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readonly');
            const store = transaction.objectStore(storeName);
            const request = store.getAll();
            
            request.onsuccess = () => resolve(request.result);
            request.onerror = () => reject(request.error);
        });
    },
    
    delete(storeName, key) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], 'readwrite');
            const store = transaction.objectStore(storeName);
            const request = store.delete(key);
            
            request.onsuccess = () => resolve(request.result);
            request.onerror = () => reject(request.error);
        });
    },
    
    // Utility
    generateGUID() {
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
            const r = Math.random() * 16 | 0;
            const v = c === 'x' ? r : (r & 0x3 | 0x8);
            return v.toString(16);
        });
    },
    
    // Export database (for backup)
    exportToJSON() {
        return Promise.all([
            this.getAll('games'),
            this.getAll('players'),
            this.getAll('roster'),
            this.getAll('chipCounts'),
            this.getAll('settings')
        ]).then(([games, players, roster, chipCounts, settings]) => {
            return {
                version: this.version,
                exportDate: new Date().toISOString(),
                games,
                players,
                roster,
                chipCounts,
                settings
            };
        });
    },
    
    // Import database (from backup)
    importFromJSON(data) {
        const promises = [];
        
        if (data.games) {
            data.games.forEach(game => promises.push(this.put('games', game)));
        }
        if (data.players) {
            data.players.forEach(player => promises.push(this.put('players', player)));
        }
        if (data.roster) {
            data.roster.forEach(player => promises.push(this.put('roster', player)));
        }
        if (data.chipCounts) {
            data.chipCounts.forEach(count => promises.push(this.put('chipCounts', count)));
        }
        if (data.settings) {
            data.settings.forEach(setting => promises.push(this.put('settings', setting)));
        }
        
        return Promise.all(promises);
    }
};
