// Chip Calculator Module
const chipCalc = {
    
    // Calculate if chip configuration is valid
    validate(chipConfig, playerCount, targetStack) {
        let stackPerPlayer = 0;
        const issues = [];
        const remaining = [];
        
        chipConfig.inventory.forEach(chip => {
            // Calculate stack value
            stackPerPlayer += chip.value * chip.perPlayer;
            
            // Calculate chips needed
            const needed = chip.perPlayer * playerCount;
            const remaining = chip.available - needed;
            
            // Check if enough chips available
            if (needed > chip.available) {
                issues.push({
                    color: chip.color,
                    needed: needed,
                    available: chip.available,
                    shortage: needed - chip.available
                });
            }
            
            remaining.push({
                color: chip.color,
                remaining: remaining
            });
        });
        
        return {
            valid: issues.length === 0 && stackPerPlayer === targetStack,
            stackPerPlayer: stackPerPlayer,
            issues: issues,
            remaining: remaining,
            stackMatch: stackPerPlayer === targetStack
        };
    },
    
    // Generate chip distribution breakdown text
    getDistributionText(chipConfig) {
        let text = 'Each player receives:\n';
        chipConfig.inventory.forEach(chip => {
            if (chip.perPlayer > 0) {
                text += `${chip.perPlayer} × ${chip.color} (${chip.value}) = ${chip.perPlayer * chip.value}\n`;
            }
        });
        return text;
    },
    
    // Calculate total chips in play
    getTotalChipsInPlay(chipConfig, playerCount) {
        let total = 0;
        chipConfig.inventory.forEach(chip => {
            total += chip.perPlayer * playerCount;
        });
        return total;
    },
    
    // Calculate total value in play
    getTotalValueInPlay(chipConfig, playerCount) {
        let total = 0;
        chipConfig.inventory.forEach(chip => {
            total += chip.value * chip.perPlayer * playerCount;
        });
        return total;
    },
    
    // Suggest optimal chip distribution for given constraints
    suggestDistribution(availableChips, playerCount, targetStack) {
        // This is a simplified suggestion algorithm
        // In practice, you might want a more sophisticated approach
        
        // Sort chips by value (descending)
        const sorted = [...availableChips].sort((a, b) => b.value - a.value);
        
        const distribution = [];
        let remainingStack = targetStack;
        
        sorted.forEach(chip => {
            const maxPerPlayer = Math.floor(chip.available / playerCount);
            const neededForStack = Math.floor(remainingStack / chip.value);
            const perPlayer = Math.min(maxPerPlayer, neededForStack);
            
            distribution.push({
                color: chip.color,
                value: chip.value,
                available: chip.available,
                perPlayer: perPlayer
            });
            
            remainingStack -= perPlayer * chip.value;
        });
        
        return {
            distribution: distribution,
            achievedStack: targetStack - remainingStack,
            perfect: remainingStack === 0
        };
    },
    
    // Calculate color-up recommendations
    getColorUpRecommendations(chipConfig, playersRemaining) {
        const recommendations = [];
        
        chipConfig.inventory.forEach((chip, index) => {
            if (index > 0) { // Skip the smallest denomination
                const prevChip = chipConfig.inventory[index - 1];
                const ratio = chip.value / prevChip.value;
                
                // If this chip is worth a lot relative to blinds, might be time to color up
                // This is a placeholder - you'd want more sophisticated logic
                recommendations.push({
                    removeColor: prevChip.color,
                    replaceWithColor: chip.color,
                    ratio: ratio,
                    suggested: false // Set based on game state
                });
            }
        });
        
        return recommendations;
    },
    
    // Format chip value for display
    formatChipValue(value) {
        if (value >= 1000000) {
            return (value / 1000000).toFixed(1) + 'M';
        } else if (value >= 1000) {
            return (value / 1000).toFixed(1) + 'K';
        }
        return value.toString();
    }
};
