// Main Application Object
const app = {
    currentScreen: 'game-running',
    currentGame: null,
    players: [],
    roster: [],
    eliminationModal: null,
    
    // Quick Start - Create a new game with default players and start immediately
    quickStart() {
        console.log('Quick Start - Creating game with defaults...');
        
        try {
            // Create default game setup
            this.currentGame = {
                id: this.generateGUID(),
                date: new Date().toISOString().split('T')[0],
                buyIn: 100,
                startingStack: 5000,
                allowRebuys: true,
                rebuyAmount: 100,
                rebuyWindowType: 'levels',
                rebuyWindowValue: 4,
                blindStructure: 'standard',
                status: 'active',
                createdDate: new Date().toISOString(),
                startedDate: new Date().toISOString(),
                chipConfig: {
                    inventory: [
                        { color: 'White', available: 300, value: 25, perPlayer: 12 },
                        { color: 'Red', available: 200, value: 100, perPlayer: 8 },
                        { color: 'Green', available: 150, value: 100, perPlayer: 4 },
                        { color: 'Blue', available: 150, value: 500, perPlayer: 5 },
                        { color: 'Black', available: 100, value: 1000, perPlayer: 1 }
                    ]
                },
                blindLevels: [
                    { level: 1, sb: 25, bb: 50, duration: 15 },
                    { level: 2, sb: 50, bb: 100, duration: 15 },
                    { level: 3, sb: 75, bb: 150, duration: 15 },
                    { level: 4, sb: 100, bb: 200, duration: 15 },
                    { level: 5, sb: 150, bb: 300, duration: 15 },
                    { level: 6, sb: 200, bb: 400, duration: 15 },
                    { level: 7, sb: 300, bb: 600, duration: 15 },
                    { level: 8, sb: 400, bb: 800, duration: 15 },
                    { level: 9, sb: 600, bb: 1200, duration: 15 },
                    { level: 10, sb: 800, bb: 1600, duration: 15 },
                    { level: 11, sb: 1000, bb: 2000, duration: 15 },
                    { level: 12, sb: 1500, bb: 3000, duration: 15 }
                ]
            };
            
            console.log('Game object created:', this.currentGame);
            
            // Create 9 default players
            this.players = [];
            for (let i = 1; i <= 9; i++) {
                this.players.push({
                    id: this.generateGUID(),
                    name: `Player ${i}`,
                    tableNumber: 1,
                    seatNumber: i,
                    buyInPaid: true,
                    rebuyCount: 0,
                    status: 'active'
                });
            }
            
            console.log('Game created with', this.players.length, 'players');
            
            // Hide overlay
            const overlay = document.getElementById('no-game-overlay');
            if (overlay) {
                overlay.style.display = 'none';
                console.log('Overlay hidden');
            } else {
                console.warn('no-game-overlay element not found');
            }
            
            // Update UI
            this.updateGameUI();
            console.log('UI updated');
            
            // Start timer
            if (window.timer) {
                console.log('Starting timer...');
                window.timer.start(this.currentGame);
                console.log('Timer started');
            } else {
                console.error('Timer module not available on window.timer');
                alert('Error: Timer module not loaded. Please refresh the page.');
                return;
            }
            
            // Save to database (async, don't wait)
            if (window.db) {
                db.saveGame(this.currentGame).catch(err => console.error('Save error:', err));
            }
            
            console.log('Quick Start complete!');
            
        } catch (error) {
            console.error('Quick Start error:', error);
            alert('Error starting game: ' + error.message);
        }
    },
    
    // Initialize the application
    init() {
        console.log('Initializing Poker Tracker...');
        
        // Setup Quick Start buttons
        const btnQuickStart = document.getElementById('btn-quick-start');
        const btnQuickStartMenu = document.getElementById('btn-quick-start-menu');
        
        if (btnQuickStart) {
            btnQuickStart.addEventListener('click', () => {
                console.log('Quick Start button clicked');
                this.quickStart();
            });
        }
        
        if (btnQuickStartMenu) {
            btnQuickStartMenu.addEventListener('click', () => {
                console.log('Quick Start menu button clicked');
                this.quickStart();
            });
        }
        
        // Set today's date in game setup
        document.getElementById('game-date').valueAsDate = new Date();
        
        // Setup rebuy toggle
        document.getElementById('allow-rebuys').addEventListener('change', (e) => {
            document.getElementById('rebuy-options').style.display = 
                e.target.checked ? 'block' : 'none';
        });
        
        // Setup form submission
        document.getElementById('form-game-setup').addEventListener('submit', (e) => {
            e.preventDefault();
            this.saveGameSetup();
            this.showScreen('registration');
        });
        
        // Initialize database
        db.init().then(() => {
            console.log('Database initialized');
            this.loadRoster();
            this.checkForPausedGame();
        });
        
        // Load blind structure preview
        this.loadBlindStructure('standard');

        // Modal + overlay wiring
        this.setupEliminationModal();
        
        // Blind structure selector
        document.getElementById('blind-structure').addEventListener('change', (e) => {
            this.loadBlindStructure(e.target.value);
        });

        // Check for offline mode
        if (window.location.protocol === 'file:') {
            const downloadBtn = document.querySelector('a[href="poker-tracker.zip"]');
            if (downloadBtn) downloadBtn.style.display = 'none';
        }
    },
    
    // Modal Management
    openModal(modalName) {
        const modal = document.getElementById(`modal-${modalName}`);
        if (modal) {
            modal.classList.add('active');
            modal.setAttribute('aria-hidden', 'false');
            
            // Initialize specific modal logic
            if (modalName === 'roster' && typeof this.initRosterScreen === 'function') this.initRosterScreen();
            if (modalName === 'history' && typeof this.initHistoryScreen === 'function') this.initHistoryScreen();
            if (modalName === 'add-player') {
                document.getElementById('form-add-player').reset();
                setTimeout(() => document.getElementById('new-player-name').focus(), 100);
            }
        }
    },

    closeModal(modalName) {
        const modal = document.getElementById(`modal-${modalName}`);
        if (modal) {
            modal.classList.remove('active');
            modal.setAttribute('aria-hidden', 'true');
        }
    },

    // Screen Management
    showScreen(screenName) {
        console.log('showScreen called with:', screenName);
        
        // Redirect to modals for specific screens
        if (['roster', 'settings', 'history'].includes(screenName)) {
            this.openModal(screenName);
            return;
        }

        // Hide all screens
        document.querySelectorAll('.screen').forEach(screen => {
            screen.classList.remove('active');
        });
        
        // Show selected screen
        const targetScreen = document.getElementById(`screen-${screenName}`);
        console.log('Target screen element:', targetScreen);
        if (targetScreen) {
            targetScreen.classList.add('active');
            this.currentScreen = screenName;
            
            // Execute screen-specific initialization
            // Convert hyphenated names to camelCase (e.g., 'chip-config' -> 'ChipConfig')
            const camelCaseName = screenName.split('-')
                .map(word => word.charAt(0).toUpperCase() + word.slice(1))
                .join('');
            const initMethod = `init${camelCaseName}Screen`;
            
            if (typeof this[initMethod] === 'function') {
                this[initMethod]();
            }
        } else {
            console.error('Screen not found:', screenName);
        }
    },
    
    // Check for paused game on startup
    checkForPausedGame() {
        db.getPausedGame().then(game => {
            if (game) {
                // document.getElementById('btn-resume').style.display = 'block'; // Button removed
                this.currentGame = game;
                console.log("Paused game found", game);
                // Optionally restore state here
                document.getElementById('no-game-overlay').style.display = 'none';
                this.resumeGame();
            } else {
                // No active game, ensure overlay is visible only if we're on game-running screen
                // and not in the middle of starting a game
                const overlay = document.getElementById('no-game-overlay');
                if (overlay && this.currentScreen === 'game-running' && !this.currentGame) {
                    overlay.style.display = 'flex';
                }
            }
        });
    },
    
    // Resume paused game
    resumeGame() {
        if (this.currentGame) {
            this.showScreen('game-running');
            
            // Hide overlay
            const overlay = document.getElementById('no-game-overlay');
            if (overlay) overlay.style.display = 'none';

            // Restore timer
            if (window.timer) {
                if (this.currentGame.timerState) {
                    timer.setState(this.currentGame.timerState);
                } else {
                    // Fallback: restart timer if state is missing
                    console.warn('No timer state found, restarting timer.');
                    timer.start(this.currentGame);
                }
            }
            
            // Restore players
            db.getPlayers(this.currentGame.id).then(players => {
                this.players = players;
                this.updateLeaderboard();
            });
        }
    },
    
    // Game Setup
    saveGameSetup() {
        const allowRebuys = document.getElementById('allow-rebuys').checked;
        const rebuyWindowType = document.querySelector('input[name="rebuy-window-type"]:checked').value;
        
        this.currentGame = {
            id: this.generateGUID(),
            date: document.getElementById('game-date').value,
            buyIn: parseFloat(document.getElementById('buy-in').value),
            startingStack: parseInt(document.getElementById('starting-stack').value),
            allowRebuys: allowRebuys,
            rebuyAmount: allowRebuys ? parseFloat(document.getElementById('rebuy-amount').value) : 0,
            rebuyWindowType: allowRebuys ? rebuyWindowType : null,
            rebuyWindowTime: allowRebuys && rebuyWindowType === 'time' ? 
                parseInt(document.getElementById('rebuy-window-time').value) : null,
            rebuyWindowLevels: allowRebuys && rebuyWindowType === 'levels' ? 
                parseInt(document.getElementById('rebuy-window-levels').value) : null,
            blindStructure: document.getElementById('blind-structure').value,
            status: 'setup',
            createdDate: new Date().toISOString()
        };
        
        console.log('Game setup saved:', this.currentGame);
    },
    
    // Load blind structure
    loadBlindStructure(structureType) {
        const preview = document.getElementById('blind-structure-preview');
        const structures = {
            standard: [
                { level: 1, sb: 25, bb: 50, duration: 15 },
                { level: 2, sb: 50, bb: 100, duration: 15 },
                { level: 3, sb: 75, bb: 150, duration: 15 },
                { level: 4, sb: 100, bb: 200, duration: 15 },
                { level: 5, sb: 150, bb: 300, duration: 15 },
                { level: 6, sb: 200, bb: 400, duration: 15 },
                { level: 7, sb: 300, bb: 600, duration: 15 },
                { level: 8, sb: 400, bb: 800, duration: 15 },
                { level: 9, sb: 600, bb: 1200, duration: 15 },
                { level: 10, sb: 800, bb: 1600, duration: 15 },
                { level: 11, sb: 1000, bb: 2000, duration: 15 },
                { level: 12, sb: 1500, bb: 3000, duration: 15 }
            ],
            deep: [
                { level: 1, sb: 25, bb: 50, duration: 20 },
                { level: 2, sb: 50, bb: 100, duration: 20 },
                { level: 3, sb: 75, bb: 150, duration: 20 },
                { level: 4, sb: 100, bb: 200, duration: 20 },
                { level: 5, sb: 150, bb: 300, duration: 20 },
                { level: 6, sb: 200, bb: 400, duration: 20 },
                { level: 7, sb: 300, bb: 600, duration: 20 },
                { level: 8, sb: 400, bb: 800, duration: 20 },
                { level: 9, sb: 600, bb: 1200, duration: 20 },
                { level: 10, sb: 800, bb: 1600, duration: 20 }
            ],
            turbo: [
                { level: 1, sb: 25, bb: 50, duration: 10 },
                { level: 2, sb: 50, bb: 100, duration: 10 },
                { level: 3, sb: 100, bb: 200, duration: 10 },
                { level: 4, sb: 150, bb: 300, duration: 10 },
                { level: 5, sb: 200, bb: 400, duration: 10 },
                { level: 6, sb: 300, bb: 600, duration: 10 },
                { level: 7, sb: 500, bb: 1000, duration: 10 },
                { level: 8, sb: 800, bb: 1600, duration: 10 }
            ]
        };
        
        const structure = structures[structureType] || structures.standard;
        
        let html = '<table style="width: 100%; margin-top: 1rem;"><thead><tr><th>Level</th><th>Small Blind</th><th>Big Blind</th><th>Duration</th></tr></thead><tbody>';
        structure.forEach(level => {
            html += `<tr>
                <td>${level.level}</td>
                <td>${level.sb}</td>
                <td>${level.bb}</td>
                <td>${level.duration} min</td>
            </tr>`;
        });
        html += '</tbody></table>';
        
        preview.innerHTML = html;
        
        // Store structure in currentGame when it exists
        if (this.currentGame) {
            this.currentGame.blindLevels = structure;
        }
    },
    
    // Player Registration
    initRegistrationScreen() {
        this.renderRoster();
        this.updateRegistrationCount();
    },
    
    loadRoster() {
        // Load roster from database
        db.getAllPlayers().then(players => {
            this.roster = players;
            if (this.currentScreen === 'registration') {
                this.renderRoster();
            }
        });
    },
    
    renderRoster() {
        const rosterList = document.getElementById('roster-list');
        
        if (this.roster.length === 0) {
            rosterList.innerHTML = '<p style="color: #888; padding: 1rem;">No players in roster. Add some in Roster Management.</p>';
            return;
        }
        
        let html = '';
        this.roster.forEach(player => {
            const isSelected = this.players.some(p => p.id === player.id);
            html += `
                <div class="player-item">
                    <input type="checkbox" 
                        id="roster-player-${player.id}" 
                        ${isSelected ? 'checked' : ''}
                        onchange="app.togglePlayer('${player.id}')">
                    <label for="roster-player-${player.id}">${player.name}</label>
                </div>
            `;
        });
        
        rosterList.innerHTML = html;
        this.renderSelectedPlayers();
    },
    
    togglePlayer(playerId) {
        const player = this.roster.find(p => p.id === playerId);
        const index = this.players.findIndex(p => p.id === playerId);
        
        if (index >= 0) {
            this.players.splice(index, 1);
        } else {
            this.players.push({
                ...player,
                tableNumber: null,
                seatNumber: null,
                buyInPaid: false,
                rebuyCount: 0,
                status: 'active'
            });
        }
        
        this.renderSelectedPlayers();
        this.updateRegistrationCount();
    },
    
    renderSelectedPlayers() {
        const selectedList = document.getElementById('selected-list');
        
        if (this.players.length === 0) {
            selectedList.innerHTML = '<p style="color: #888; padding: 1rem;">No players selected</p>';
            return;
        }
        
        let html = '';
        this.players.forEach(player => {
            html += `
                <div class="player-item">
                    <span>${player.name}</span>
                    <label>
                        <input type="checkbox" 
                            ${player.buyInPaid ? 'checked' : ''}
                            onchange="app.toggleBuyInPaid('${player.id}')">
                        Paid
                    </label>
                </div>
            `;
        });
        
        selectedList.innerHTML = html;
    },
    
    toggleBuyInPaid(playerId) {
        const player = this.players.find(p => p.id === playerId);
        if (player) {
            player.buyInPaid = !player.buyInPaid;
        }
    },
    
    updateRegistrationCount() {
        document.getElementById('registered-count').textContent = this.players.length;
    },
    
    addWalkIn() {
        const name = prompt('Enter player name:');
        if (name && name.trim()) {
            const newPlayer = {
                id: this.generateGUID(),
                name: name.trim(),
                tableNumber: null,
                seatNumber: null,
                buyInPaid: false,
                rebuyCount: 0,
                status: 'active',
                isWalkIn: true
            };
            this.players.push(newPlayer);
            this.renderSelectedPlayers();
            this.updateRegistrationCount();
        }
    },
    
    // Chip Configuration
    goToChipConfig() {
        if (this.players.length === 0) {
            alert('Please select at least one player first.');
            return;
        }
        
        // Initialize chip config if not exists
        if (!this.currentGame.chipConfig) {
            this.currentGame.chipConfig = {
                inventory: [
                    { color: 'White', available: 300, value: 25, perPlayer: 12 },
                    { color: 'Red', available: 200, value: 100, perPlayer: 8 },
                    { color: 'Green', available: 150, value: 100, perPlayer: 4 },
                    { color: 'Blue', available: 150, value: 500, perPlayer: 5 },
                    { color: 'Black', available: 100, value: 1000, perPlayer: 1 }
                ]
            };
        }
        
        this.showScreen('chip-config');
    },
    
    initChipConfigScreen() {
        // Initialize with default chip setup
        if (!this.currentGame.chipConfig) {
            this.currentGame.chipConfig = {
                inventory: [
                    { color: 'White', available: 300, value: 25, perPlayer: 12 },
                    { color: 'Red', available: 200, value: 100, perPlayer: 8 },
                    { color: 'Green', available: 150, value: 100, perPlayer: 4 },
                    { color: 'Blue', available: 150, value: 500, perPlayer: 5 },
                    { color: 'Black', available: 100, value: 1000, perPlayer: 1 }
                ]
            };
        }
        this.renderChipConfig();
        this.validateChipConfig();
    },
    
    renderChipConfig() {
        const inventoryList = document.getElementById('chip-inventory-list');
        const distributionList = document.getElementById('chip-distribution-list');
        
        let inventoryHTML = '';
        let distributionHTML = '';
        
        this.currentGame.chipConfig.inventory.forEach((chip, index) => {
            inventoryHTML += `
                <div class="chip-row">
                    <input type="text" value="${chip.color}" 
                        onchange="app.updateChipColor(${index}, this.value)">
                    <input type="number" value="${chip.available}" min="0"
                        onchange="app.updateChipAvailable(${index}, this.value)">
                    <button class="btn-danger btn-small" onclick="app.removeChipColor(${index})" style="padding: 0.5rem;">×</button>
                </div>
            `;
            
            distributionHTML += `
                <div class="chip-row">
                    <span>${chip.color}</span>
                    <input type="number" value="${chip.value}" min="1"
                        onchange="app.updateChipValue(${index}, this.value)" placeholder="Value">
                    <input type="number" value="${chip.perPlayer}" min="0"
                        onchange="app.updateChipPerPlayer(${index}, this.value)" placeholder="Per Player">
                </div>
            `;
        });
        
        inventoryList.innerHTML = inventoryHTML;
        distributionList.innerHTML = distributionHTML;
    },
    
    updateChipColor(index, value) {
        this.currentGame.chipConfig.inventory[index].color = value;
        this.renderChipConfig();
    },
    
    updateChipAvailable(index, value) {
        this.currentGame.chipConfig.inventory[index].available = parseInt(value);
        this.validateChipConfig();
    },
    
    updateChipValue(index, value) {
        this.currentGame.chipConfig.inventory[index].value = parseInt(value);
        this.validateChipConfig();
    },
    
    updateChipPerPlayer(index, value) {
        this.currentGame.chipConfig.inventory[index].perPlayer = parseInt(value);
        this.validateChipConfig();
    },
    
    addChipColor() {
        // Ensure chipConfig exists
        if (!this.currentGame || !this.currentGame.chipConfig) {
            console.error('Chip config not initialized');
            alert('Please complete game setup first');
            return;
        }
        
        const color = prompt('Enter chip color name:');
        if (color && color.trim()) {
            console.log('Adding chip color:', color.trim());
            this.currentGame.chipConfig.inventory.push({
                color: color.trim(),
                available: 100,
                value: 100,
                perPlayer: 0
            });
            console.log('Current inventory:', this.currentGame.chipConfig.inventory);
            this.renderChipConfig();
            this.validateChipConfig();
        }
    },
    
    removeChipColor(index) {
        if (confirm(`Remove ${this.currentGame.chipConfig.inventory[index].color} chips?`)) {
            this.currentGame.chipConfig.inventory.splice(index, 1);
            this.renderChipConfig();
            this.validateChipConfig();
        }
    },
    
    validateChipConfig() {
        // Check if chipConfig exists, if not initialize with defaults
        if (!this.currentGame || !this.currentGame.chipConfig || !this.currentGame.chipConfig.inventory) {
            console.log('Chip config not initialized, creating defaults...');
            if (!this.currentGame) {
                console.error('No current game exists');
                return false;
            }
            this.currentGame.chipConfig = {
                inventory: [
                    { color: 'White', available: 300, value: 25, perPlayer: 12 },
                    { color: 'Red', available: 200, value: 100, perPlayer: 8 },
                    { color: 'Green', available: 150, value: 100, perPlayer: 4 },
                    { color: 'Blue', available: 150, value: 500, perPlayer: 5 },
                    { color: 'Black', available: 100, value: 1000, perPlayer: 1 }
                ]
            };
        }
        
        const playerCount = this.players.length;
        let stackPerPlayer = 0;
        let allValid = true;
        let validationMessages = [];
        let chipDetails = [];
        
        console.log('Validating chip config for', playerCount, 'players');
        console.log('Current inventory:', this.currentGame.chipConfig.inventory);
        
        this.currentGame.chipConfig.inventory.forEach(chip => {
            stackPerPlayer += chip.value * chip.perPlayer;
            const needed = chip.perPlayer * playerCount;
            const remaining = chip.available - needed;
            
            chipDetails.push({
                color: chip.color,
                needed: needed,
                available: chip.available,
                remaining: remaining
            });
            
            if (needed > chip.available) {
                allValid = false;
                validationMessages.push(`${chip.color}: Need ${needed}, only ${chip.available} available`);
            }
        });
        
        console.log('Stack per player:', stackPerPlayer, 'Target:', this.currentGame.startingStack);
        console.log('Chip details:', chipDetails);
        
        // Update stack per player display
        const stackElement = document.getElementById('stack-per-player');
        if (stackElement) {
            stackElement.textContent = stackPerPlayer.toLocaleString();
            stackElement.style.color = 
                stackPerPlayer === this.currentGame.startingStack ? '#4CAF50' : '#f44336';
        }
        
        // Update total chips needed
        const neededElement = document.getElementById('chips-needed');
        if (neededElement) {
            let chipsNeededHTML = '<div style="font-size: 0.9rem; margin-top: 0.5rem;">';
            chipDetails.forEach(detail => {
                const color = detail.needed > detail.available ? '#f44336' : 
                             detail.remaining < 10 ? '#ff9800' : '#4CAF50';
                chipsNeededHTML += `<div style="display: flex; justify-content: space-between; padding: 0.25rem;">
                    <span>${detail.color}:</span>
                    <span style="color: ${color};">${detail.needed} / ${detail.available}</span>
                </div>`;
            });
            chipsNeededHTML += '</div>';
            neededElement.innerHTML = chipsNeededHTML;
        } else {
            console.error('chips-needed element not found');
        }
        
        // Update chips remaining
        const remainingElement = document.getElementById('chips-remaining');
        if (remainingElement) {
            let chipsRemainingHTML = '<div style="font-size: 0.9rem; margin-top: 0.5rem;">';
            chipDetails.forEach(detail => {
                const color = detail.remaining < 0 ? '#f44336' : 
                             detail.remaining < 10 ? '#ff9800' : '#4CAF50';
                chipsRemainingHTML += `<div style="display: flex; justify-content: space-between; padding: 0.25rem;">
                    <span>${detail.color}:</span>
                    <span style="color: ${color};">${detail.remaining}</span>
                </div>`;
            });
            chipsRemainingHTML += '</div>';
            remainingElement.innerHTML = chipsRemainingHTML;
        } else {
            console.error('chips-remaining element not found');
        }
        
        // Update status message
        const statusDiv = document.getElementById('validation-status');
        if (statusDiv) {
            if (!allValid) {
                statusDiv.className = 'status-message error';
                statusDiv.textContent = '⚠ ' + validationMessages.join('; ');
            } else if (stackPerPlayer !== this.currentGame.startingStack) {
                statusDiv.className = 'status-message error';
                statusDiv.textContent = `⚠ Stack per player (${stackPerPlayer.toLocaleString()}) doesn't match starting stack (${this.currentGame.startingStack.toLocaleString()})`;
            } else {
                statusDiv.className = 'status-message success';
                statusDiv.textContent = '✓ Configuration valid - Ready to start!';
            }
        }
        
        return allValid && stackPerPlayer === this.currentGame.startingStack;
    },
    
    // Start Game
    startGame() {
        console.log('startGame() called');
        
        if (!this.currentGame) {
            alert('Error: No game configuration found. Please complete game setup first.');
            this.showScreen('game-setup');
            return;
        }
        
        if (!this.players || this.players.length === 0) {
            alert('Error: No players registered. Please add players before starting.');
            this.showScreen('registration');
            return;
        }
        
        if (!this.validateChipConfig()) {
            alert('Please fix chip configuration errors before starting.');
            return;
        }
        
        // Assign random seating
        this.assignSeating();
        
        // Load blind structure
        const structures = {
            standard: [
                { level: 1, sb: 25, bb: 50, duration: 15 },
                { level: 2, sb: 50, bb: 100, duration: 15 },
                { level: 3, sb: 75, bb: 150, duration: 15 },
                { level: 4, sb: 100, bb: 200, duration: 15 },
                { level: 5, sb: 150, bb: 300, duration: 15 },
                { level: 6, sb: 200, bb: 400, duration: 15 },
                { level: 7, sb: 300, bb: 600, duration: 15 },
                { level: 8, sb: 400, bb: 800, duration: 15 },
                { level: 9, sb: 600, bb: 1200, duration: 15 },
                { level: 10, sb: 800, bb: 1600, duration: 15 },
                { level: 11, sb: 1000, bb: 2000, duration: 15 },
                { level: 12, sb: 1500, bb: 3000, duration: 15 }
            ],
            deep: [
                { level: 1, sb: 25, bb: 50, duration: 20 },
                { level: 2, sb: 50, bb: 100, duration: 20 },
                { level: 3, sb: 75, bb: 150, duration: 20 },
                { level: 4, sb: 100, bb: 200, duration: 20 },
                { level: 5, sb: 150, bb: 300, duration: 20 },
                { level: 6, sb: 200, bb: 400, duration: 20 },
                { level: 7, sb: 300, bb: 600, duration: 20 },
                { level: 8, sb: 400, bb: 800, duration: 20 },
                { level: 9, sb: 600, bb: 1200, duration: 20 },
                { level: 10, sb: 800, bb: 1600, duration: 20 }
            ],
            turbo: [
                { level: 1, sb: 25, bb: 50, duration: 10 },
                { level: 2, sb: 50, bb: 100, duration: 10 },
                { level: 3, sb: 100, bb: 200, duration: 10 },
                { level: 4, sb: 150, bb: 300, duration: 10 },
                { level: 5, sb: 200, bb: 400, duration: 10 },
                { level: 6, sb: 300, bb: 600, duration: 10 },
                { level: 7, sb: 500, bb: 1000, duration: 10 },
                { level: 8, sb: 800, bb: 1600, duration: 10 }
            ]
        };
        
        this.currentGame.blindLevels = structures[this.currentGame.blindStructure] || structures.standard;
        
        // Save game to database
        this.currentGame.status = 'active';
        this.currentGame.startedDate = new Date().toISOString();
        
        // Force UI update immediately to prevent "No Game Started" lag
        this.showScreen('game-running');
        
        // Hide overlay immediately and ensure it stays hidden
        const overlay = document.getElementById('no-game-overlay');
        if (overlay) {
            overlay.style.display = 'none';
            overlay.style.visibility = 'hidden';
        }
        
        console.log('Game starting with blind structure:', this.currentGame.blindLevels);

        // Initialize timer immediately
        if (window.timer) {
            timer.start(this.currentGame);
            // Update UI immediately after timer starts
            this.updateGameUI();
        } else {
            console.error('Timer module not loaded');
            alert('Error: Timer module failed to load. Please refresh the page.');
        }

        db.saveGame(this.currentGame).then(() => {
            db.savePlayers(this.currentGame.id, this.players).then(() => {
                console.log('Game started and saved:', this.currentGame);
            }).catch(err => {
                console.error('Failed to save players:', err);
                alert('Game started but failed to save player data. Please check console.');
            });
        }).catch(err => {
            console.error('Failed to save game:', err);
            alert('Game started but failed to save to database. Local storage might be full.');
        });
    },

    // Factory Reset
    factoryReset() {
        if (confirm('Are you sure you want to wipe all data? This cannot be undone.')) {
            localStorage.clear();
            window.location.reload();
        }
    },
    
    assignSeating() {
        // Shuffle players
        const shuffled = [...this.players].sort(() => Math.random() - 0.5);
        
        // Assign to tables (9-10 per table)
        const playersPerTable = Math.ceil(shuffled.length / 3);
        let currentTable = 1;
        let currentSeat = 1;
        
        shuffled.forEach((player, index) => {
            player.tableNumber = currentTable;
            player.seatNumber = currentSeat;
            
            currentSeat++;
            if (currentSeat > playersPerTable && currentTable < 3) {
                currentTable++;
                currentSeat = 1;
            }
        });
        
        console.log('Seating assigned:', this.players);
    },
    
    // Utility
    generateGUID() {
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
            const r = Math.random() * 16 | 0;
            const v = c === 'x' ? r : (r & 0x3 | 0x8);
            return v.toString(16);
        });
    },
    
    // Tab switching for game screen
    switchTab(tabName) {
        document.querySelectorAll('.tab-button').forEach(btn => btn.classList.remove('active'));
        document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
        
        event.target.classList.add('active');
        document.getElementById(`tab-${tabName}`).classList.add('active');
    },
    
    // Toggle admin panels (collapsible sections)
    toggleAdminPanel(panelName) {
        const content = document.getElementById(`panel-${panelName}`);
        const icon = document.getElementById(`toggle-${panelName}`);
        
        if (content.style.display === 'none') {
            content.style.display = 'block';
            icon.classList.remove('collapsed');
        } else {
            content.style.display = 'none';
            icon.classList.add('collapsed');
        }
    },
    
    // Initialize game running screen
    initGameRunningScreen() {
        // Calculate and display table status
        this.updateTableStatus();
        
        // Update rebuy status
        this.updateRebuyStatus();
        
        // Show blind structure
        this.displayBlindStructure();
        
        // Start updating stats
        this.updateGameStats();
    },
    
    updateTableStatus() {
        const tables = {};
        this.players.filter(p => p.status === 'active').forEach(player => {
            if (!tables[player.tableNumber]) {
                tables[player.tableNumber] = [];
            }
            tables[player.tableNumber].push(player);
        });
        
        let html = '';
        Object.keys(tables).sort().forEach(tableNum => {
            const players = tables[tableNum];
            html += `<div style="padding: 0.5rem; background: #252525; border-radius: 4px; margin-bottom: 0.5rem;">
                <strong>Table ${tableNum}:</strong> ${players.length} players
            </div>`;
        });
        
        document.getElementById('table-status').innerHTML = html;
    },
    
    updateRebuyStatus() {
        if (!this.currentGame.allowRebuys) {
            document.getElementById('rebuy-panel-container').style.display = 'none';
            return;
        }
        
        // TODO: Calculate if rebuy window is still open
        const statusText = document.getElementById('rebuy-status-text');
        statusText.textContent = 'OPEN';
        statusText.style.color = '#4CAF50';
    },
    
    displayBlindStructure() {
        if (!this.currentGame || !this.currentGame.blindLevels) return;
        
        let html = '<table style="width: 100%; font-size: 0.9rem;"><thead><tr><th>Lvl</th><th>SB/BB</th><th>Time</th></tr></thead><tbody>';
        this.currentGame.blindLevels.forEach((level, index) => {
            const isCurrent = timer.currentLevel === index + 1;
            html += `<tr style="background: ${isCurrent ? '#4CAF50' : 'transparent'}; color: ${isCurrent ? '#000' : '#e0e0e0'};">
                <td>${level.level}</td>
                <td>${level.sb}/${level.bb}</td>
                <td>${level.duration}m</td>
            </tr>`;
        });
        html += '</tbody></table>';
        
        document.getElementById('blind-structure-list').innerHTML = html;
    },
    
    updateGameStats() {
        const activePlayers = this.players.filter(p => p.status === 'active').length;
        const eliminatedPlayers = this.players.filter(p => p.status === 'eliminated').length;
        const totalRebuys = this.players.reduce((sum, p) => sum + (p.rebuyCount || 0), 0);
        const prizePool = (this.players.length * this.currentGame.buyIn) + (totalRebuys * this.currentGame.rebuyAmount);
        
        // Update multiple locations
        document.getElementById('active-players').textContent = activePlayers;
        document.getElementById('eliminated-players').textContent = eliminatedPlayers;
        document.getElementById('rebuy-count').textContent = totalRebuys;
        document.getElementById('prize-pool').textContent = `$${prizePool.toLocaleString()}`;
        
        document.getElementById('summary-prize-pool').textContent = `$${prizePool.toLocaleString()}`;
        document.getElementById('summary-players-remaining').textContent = activePlayers;
        
        // Calculate average stack
        const totalChips = activePlayers * this.currentGame.startingStack;
        const avgStack = activePlayers > 0 ? Math.floor(totalChips / activePlayers) : 0;
        document.getElementById('summary-avg-stack').textContent = avgStack.toLocaleString();
    },
    
    // Update Game UI - synchronize all display elements
    updateGameUI() {
        if (!this.currentGame || !this.currentGame.blindLevels) {
            console.warn('Game or blind structure not initialized');
            return;
        }

        // Update stats
        const activePlayers = this.players.filter(p => p.status === 'active').length;
        const totalRebuys = this.players.reduce((sum, p) => sum + (p.rebuyCount || 0), 0);
        const prizePool = (this.players.length * this.currentGame.buyIn) + (totalRebuys * this.currentGame.rebuyAmount);
        
        document.getElementById('summary-prize-pool').textContent = `$${prizePool.toLocaleString()}`;
        document.getElementById('summary-players-remaining').textContent = activePlayers;
        
        const totalChips = activePlayers * this.currentGame.startingStack;
        const avgStack = activePlayers > 0 ? Math.floor(totalChips / activePlayers) : 0;
        document.getElementById('summary-avg-stack').textContent = avgStack.toLocaleString();

        // Update player counts
        document.getElementById('active-players').textContent = activePlayers;
        document.getElementById('eliminated-players').textContent = this.players.filter(p => p.status === 'eliminated').length;
        document.getElementById('rebuy-count').textContent = totalRebuys;
        document.getElementById('prize-pool').textContent = `$${prizePool.toLocaleString()}`;

        console.log('Game UI updated:', { activePlayers, prizePool, avgStack });
    },

    // Placeholder methods for game running
    togglePause() {
        timer.togglePause();
        const btnText = document.getElementById('pause-resume-text');
        if (btnText) {
            btnText.textContent = timer.isPaused ? 'Resume' : 'Pause';
        }
        
        // Also update the button style in app.js to match timer.js logic if needed, 
        // although timer.js handles the button styling directly.
    },
    
    adjustTime(minutes) {
        timer.adjustTime(minutes);
    },
    
    skipLevel() {
        timer.skipLevel();
    },
    
    setupEliminationModal() {
        const overlay = document.getElementById('modal-eliminate-player');
        if (!overlay) {
            console.warn('Elimination modal markup missing');
            return;
        }
        this.eliminationModal = {
            overlay,
            list: document.getElementById('elimination-player-list'),
            search: document.getElementById('elimination-search'),
            place: document.getElementById('elimination-place'),
            notes: document.getElementById('elimination-notes'),
            level: document.getElementById('elimination-level'),
            clock: document.getElementById('elimination-clock'),
            remaining: document.getElementById('elimination-remaining'),
            confirm: document.getElementById('btn-confirm-elimination'),
            cancel: document.getElementById('btn-cancel-elimination'),
            close: document.getElementById('btn-close-elimination'),
            selectedPlayerId: null
        };
        const modal = this.eliminationModal;
        if (modal.search) {
            modal.search.addEventListener('input', (e) => this.renderEliminationList(e.target.value));
        }
        if (modal.list) {
            modal.list.addEventListener('change', (e) => {
                if (e.target && e.target.name === 'elimination-player') {
                    modal.selectedPlayerId = e.target.value;
                }
            });
        }
        modal.confirm?.addEventListener('click', () => this.confirmElimination());
        modal.cancel?.addEventListener('click', () => this.closeEliminationModal());
        modal.close?.addEventListener('click', () => this.closeEliminationModal());
        overlay.addEventListener('click', (e) => {
            if (e.target === overlay) {
                this.closeEliminationModal();
            }
        });
    },
    
    renderEliminationList(filterTerm = '') {
        if (!this.eliminationModal || !this.eliminationModal.list) return;
        const listEl = this.eliminationModal.list;
        const searchTerm = filterTerm.trim().toLowerCase();
        const activePlayers = this.players.filter(p => (p.status || 'active') === 'active');
        if (activePlayers.length === 0) {
            listEl.innerHTML = '<p class="empty-state">All players have been eliminated.</p>';
            return;
        }
        const filtered = activePlayers
            .filter(player => {
                if (!searchTerm) return true;
                const haystack = `${player.name} table ${player.tableNumber || ''} seat ${player.seatNumber || ''}`.toLowerCase();
                return haystack.includes(searchTerm);
            })
            .sort((a, b) => {
                const tableDiff = (a.tableNumber || 0) - (b.tableNumber || 0);
                if (tableDiff !== 0) return tableDiff;
                return (a.seatNumber || 0) - (b.seatNumber || 0);
            });
        if (filtered.length === 0) {
            listEl.innerHTML = '<p class="empty-state">No players match that search.</p>';
            return;
        }
        const selectedId = this.eliminationModal.selectedPlayerId;
        let html = '';
        filtered.forEach((player, index) => {
            const tableLabel = player.tableNumber ? `Table ${player.tableNumber}` : 'Unassigned';
            const seatLabel = player.seatNumber ? `Seat ${player.seatNumber}` : '';
            const meta = [tableLabel, seatLabel].filter(Boolean).join(' · ');
            const stackValue = typeof player.stack === 'number' ? `${player.stack.toLocaleString()} chips` : 'Stack N/A';
            const shouldCheck = selectedId ? selectedId === player.id : index === 0;
            html += `
                <label class="player-radio-row">
                    <input type="radio" name="elimination-player" value="${player.id}" ${shouldCheck ? 'checked' : ''}>
                    <div>
                        <div class="player-name">${player.name}</div>
                        <div class="player-meta">${meta}</div>
                    </div>
                    <span class="player-stack">${stackValue}</span>
                </label>
            `;
        });
        listEl.innerHTML = html;
        if (!selectedId && filtered.length > 0) {
            this.eliminationModal.selectedPlayerId = filtered[0].id;
        }
    },
    
    openEliminationModal() {
        if (!this.eliminationModal) return;
        const modal = this.eliminationModal;
        modal.overlay.classList.add('open');
        modal.overlay.setAttribute('aria-hidden', 'false');
        if (modal.search) {
            modal.search.value = '';
        }
        if (modal.notes) {
            modal.notes.value = '';
        }
        modal.selectedPlayerId = null;
        const defaultPlace = this.getDefaultEliminationPlace();
        if (modal.place) {
            modal.place.value = defaultPlace || '';
        }
        if (modal.level) {
            modal.level.textContent = timer.currentLevel || 1;
        }
        const clockDisplay = document.getElementById('time-remaining');
        if (modal.clock) {
            modal.clock.textContent = clockDisplay ? clockDisplay.textContent : '—';
        }
        if (modal.remaining) {
            modal.remaining.textContent = this.players.filter(p => (p.status || 'active') === 'active').length;
        }
        this.renderEliminationList();
    },
    
    closeEliminationModal() {
        if (!this.eliminationModal) return;
        this.eliminationModal.overlay.classList.remove('open');
        this.eliminationModal.overlay.setAttribute('aria-hidden', 'true');
        this.eliminationModal.selectedPlayerId = null;
    },
    
    getDefaultEliminationPlace() {
        const totalPlayers = this.players.length;
        const eliminatedPlayers = this.players.filter(p => p.status === 'eliminated').length;
        const place = totalPlayers - eliminatedPlayers;
        return place > 0 ? place : '';
    },
    
    confirmElimination() {
        if (!this.eliminationModal || !this.eliminationModal.list) return;
        const selectedInput = this.eliminationModal.list.querySelector('input[name="elimination-player"]:checked');
        if (!selectedInput) {
            alert('Please select a player to eliminate.');
            return;
        }
        const player = this.players.find(p => p.id === selectedInput.value);
        if (!player) {
            alert('Selected player could not be found.');
            return;
        }
        const placeField = this.eliminationModal.place;
        const placeValue = placeField ? parseInt(placeField.value, 10) : NaN;
        const notesField = this.eliminationModal.notes;
        const notes = notesField ? notesField.value.trim() : '';
        const eliminationRecord = {
            playerId: player.id,
            name: player.name,
            place: Number.isFinite(placeValue) ? placeValue : null,
            level: timer.currentLevel || 1,
            timeRemaining: (document.getElementById('time-remaining') || {}).textContent || null,
            timestamp: new Date().toISOString(),
            notes
        };
        player.status = 'eliminated';
        player.finishPlace = eliminationRecord.place;
        player.eliminationNotes = notes;
        player.eliminationLevel = eliminationRecord.level;
        if (!this.currentGame.eliminations) {
            this.currentGame.eliminations = [];
        }
        this.currentGame.eliminations.push(eliminationRecord);
        this.updateGameStats();
        this.updateTableStatus();
        this.closeEliminationModal();
    },
    
    eliminatePlayer() {
        if (!this.eliminationModal) {
            alert('Elimination modal is not ready yet. Try reloading the page.');
            return;
        }
        const activePlayers = this.players.filter(p => (p.status || 'active') === 'active');
        if (activePlayers.length === 0) {
            alert('All players have already been eliminated.');
            return;
        }
        this.openEliminationModal();
    },
    
    movePlayer() {
        const activePlayers = this.players.filter(p => p.status === 'active');
        if (activePlayers.length === 0) {
            alert('No active players to move.');
            return;
        }

        // Simple implementation: Prompt for player name and new table/seat
        const playerNames = activePlayers.map((p, i) => `${i + 1}. ${p.name} (T${p.tableNumber} S${p.seatNumber})`).join('\n');
        const selection = prompt(`Enter number of player to move:\n\n${playerNames}`);
        
        if (!selection) return;
        
        const index = parseInt(selection) - 1;
        if (isNaN(index) || index < 0 || index >= activePlayers.length) {
            alert('Invalid selection');
            return;
        }

        const player = activePlayers[index];
        const newTable = prompt(`Enter new Table number for ${player.name}:`, player.tableNumber);
        const newSeat = prompt(`Enter new Seat number for ${player.name}:`, player.seatNumber);

        if (newTable && newSeat) {
            player.tableNumber = parseInt(newTable);
            player.seatNumber = parseInt(newSeat);
            this.updateTableStatus();
            alert(`${player.name} moved to Table ${newTable}, Seat ${newSeat}`);
        }
    },
    
    updateTop3() {
        alert('Quick Update Top 3: Feature coming soon. Please use "Eliminate Player" to track finishing positions.');
    },
    
    updateAllStacks() {
        alert('Update All Stacks: Feature coming soon.');
    },
    
    recordRebuy() {
        const activePlayers = this.players.filter(p => p.status === 'active');
        if (activePlayers.length === 0) {
            alert('No active players.');
            return;
        }

        const playerNames = activePlayers.map((p, i) => `${i + 1}. ${p.name} (Rebuys: ${p.rebuyCount || 0})`).join('\n');
        const selection = prompt(`Select player for REBUY ($${this.currentGame.rebuyAmount}):\n\n${playerNames}`);
        
        if (!selection) return;
        
        const index = parseInt(selection) - 1;
        if (isNaN(index) || index < 0 || index >= activePlayers.length) {
            alert('Invalid selection');
            return;
        }

        const player = activePlayers[index];
        if (confirm(`Confirm rebuy for ${player.name} for $${this.currentGame.rebuyAmount}?`)) {
            player.rebuyCount = (player.rebuyCount || 0) + 1;
            this.updateGameStats();
            alert(`Rebuy recorded for ${player.name}. Total rebuys: ${player.rebuyCount}`);
        }
    },
    
    pauseAndSaveGame() {
        timer.togglePause();
        this.currentGame.status = 'paused';
        db.saveGame(this.currentGame);
        alert('Game paused and saved.');
        // Stay on screen or go to setup? Going to setup allows resuming later.
        this.showScreen('game-setup'); 
    },
    
    endGame() {
        if (confirm('Are you sure you want to end this game?')) {
            timer.stopCountdown();
            // TODO: Show results screen with final standings
            console.log('End game - TODO: implement results');
            this.showScreen('results');
        }
    },
    
    syncToRosterDB() {
        console.log('Sync to roster DB');
    },
    
    // Add sample players for testing
    addSamplePlayers() {
        const sampleNames = [
            'Player 1', 'Player 2', 'Player 3',
            'Sarah Johnson', 'Mike Chen', 'John Smith', 'Emily Davis',
            'David Wilson', 'Lisa Martinez', 'Robert Brown', 'Jennifer Lee',
            'Michael Taylor', 'Jessica Anderson', 'Chris Thompson', 'Amanda White',
            'Daniel Garcia', 'Ashley Martinez', 'Matthew Robinson', 'Michelle Clark',
            'James Rodriguez', 'Laura Lewis', 'Ryan Walker', 'Nicole Hall'
        ];
        
        const promises = sampleNames.map(name => {
            const player = {
                id: this.generateGUID(),
                name: name,
                email: '',
                phone: '',
                notes: 'Sample player',
                totalWinnings: 0,
                gamesPlayed: 0,
                createdDate: new Date().toISOString()
            };
            return db.addToRoster(player);
        });
        
        Promise.all(promises).then(() => {
            alert('Sample players added to roster.');
            this.loadRoster();
            if (this.currentScreen === 'roster') {
                this.renderRosterManagement();
            }
        });
    },

    initRosterScreen() {
        this.renderRosterManagement();
    },
    
    renderRosterManagement() {
        const list = document.getElementById('roster-management-list');
        
        if (this.roster.length === 0) {
            list.innerHTML = `
                <div class="roster-empty">
                    <div class="roster-empty-icon">👥</div>
                    <h3>No Players Yet</h3>
                    <p>Click "Add New Player" to start building your roster</p>
                </div>
            `;
            document.getElementById('roster-total-count').textContent = '0 players';
            return;
        }
        
        // Apply search filter if exists
        const searchTerm = document.getElementById('roster-management-search')?.value.toLowerCase() || '';
        const filteredRoster = searchTerm ? 
            this.roster.filter(p => p.name.toLowerCase().includes(searchTerm)) : 
            this.roster;
        
        let html = '';
        filteredRoster.forEach(player => {
            const initials = player.name.split(' ').map(n => n[0]).join('').toUpperCase().slice(0, 2);
            const photoDisplay = player.photo ? 
                `<img src="${player.photo}" alt="${player.name}">` : 
                initials;
            
            html += `
                <div class="roster-card">
                    <div class="roster-card-header">
                        <div class="roster-card-photo">${photoDisplay}</div>
                        <div class="roster-card-info">
                            <div class="roster-card-name">${player.name}</div>
                            <div class="roster-card-stats">
                                ${player.gamesPlayed || 0} games • 
                                ${player.totalWinnings ? '$' + player.totalWinnings.toLocaleString() : '$0'}
                            </div>
                        </div>
                    </div>
                    <div class="roster-card-actions">
                        <button class="btn-secondary btn-small" onclick="app.editPlayer('${player.id}')">Edit</button>
                        <button class="btn-danger btn-small" onclick="app.deletePlayer('${player.id}')">Delete</button>
                    </div>
                </div>
            `;
        });
        
        list.innerHTML = html;
        document.getElementById('roster-total-count').textContent = 
            `${filteredRoster.length} ${filteredRoster.length === this.roster.length ? '' : 'of ' + this.roster.length} player${this.roster.length === 1 ? '' : 's'}`;
    },
    
    filterRosterList() {
        this.renderRosterManagement();
    },

    saveNewPlayer(event) {
        event.preventDefault();
        const nameInput = document.getElementById('new-player-name');
        const emailInput = document.getElementById('new-player-email');
        const phoneInput = document.getElementById('new-player-phone');
        
        const name = nameInput.value.trim();
        if (!name) return;
        
        const player = {
            id: this.generateGUID(),
            name: name,
            email: emailInput.value.trim(),
            phone: phoneInput.value.trim(),
            photo: null,
            gamesPlayed: 0,
            totalWinnings: 0,
            bestFinish: null,
            lastPlayed: null,
            createdDate: new Date().toISOString()
        };
        
        db.addToRoster(player).then(() => {
            console.log('Player added:', player);
            this.loadRoster();
            this.renderRosterManagement();
            this.closeModal('add-player');
        });
    },

    addNewPlayer() {
        // Deprecated in favor of openModal('add-player')
        this.openModal('add-player');
    },

    editPlayer(playerId) {
        const player = this.roster.find(p => p.id === playerId);
        if (!player) return;
        
        const newName = prompt('Edit player name:', player.name);
        if (newName && newName.trim() && newName.trim() !== player.name) {
            player.name = newName.trim();
            db.addToRoster(player).then(() => {
                console.log('Player updated:', player);
                this.loadRoster();
                this.renderRosterManagement();
            });
        }
    },
    
    deletePlayer(playerId) {
        const player = this.roster.find(p => p.id === playerId);
        if (!player) return;
        
        if (confirm(`Delete ${player.name} from roster?\n\nThis cannot be undone.`)) {
            db.removeFromRoster(playerId).then(() => {
                console.log('Player deleted:', player);
                this.roster = this.roster.filter(p => p.id !== playerId);
                this.renderRosterManagement();
            });
        }
    },
    
    importRosterFromGSheets() {
        alert('Google Sheets import - Coming soon!\\n\\nThis will let you import your player roster from a Google Sheet.');
    },
    
    exportRosterToGSheets() {
        alert('Google Sheets export - Coming soon!\\n\\nThis will export your roster to a Google Sheet for backup or sharing.');
    },
    
    saveSettings() {
        const settings = {
            defaultBuyIn: document.getElementById('setting-default-buyin').value,
            defaultStack: document.getElementById('setting-default-stack').value,
            gsheetUrl: document.getElementById('setting-gsheet-url').value
        };
        localStorage.setItem('poker-clock-settings', JSON.stringify(settings));
        
        // Apply defaults immediately if on setup screen
        const buyInInput = document.getElementById('buy-in');
        const stackInput = document.getElementById('starting-stack');
        if (buyInInput) buyInInput.value = settings.defaultBuyIn;
        if (stackInput) stackInput.value = settings.defaultStack;
        
        alert('Settings saved!');
        this.closeModal('settings');
    },
    
    testGSheetConnection() {
        console.log('Test GSheet connection');
    },
    
    manageBlindTemplates() {
        console.log('Manage blind templates');
    },
    
    manageChipTemplates() {
        console.log('Manage chip templates');
    }
};

// Initialize on page load
document.addEventListener('DOMContentLoaded', () => {
    app.init();
});

// Global quick start function (for onclick)
window.startQuickGame = function() {
    console.log('startQuickGame called - delegating to app.quickStart()');
    app.quickStart();
};
